#include "stdafx.h"
#include "ttd.h"
#include "viewport.h"
#include "window.h"
#include "vehicle.h"
#include "station.h"
#include "gfx.h"
#include "city.h"

#define VIEWPORT_DRAW_MEM (65536)

static bool _added_tile_sprite;

typedef struct StringSpriteToDraw {
	uint16 string;
	uint16 color;
	struct StringSpriteToDraw *next;
	int16 x;
	int16 y;
	uint32 params[2];
	uint16 width;
} StringSpriteToDraw;

typedef struct TileSpriteToDraw {
	uint32 image;
	struct TileSpriteToDraw *next;
	int16 x, y;
	byte z;
} TileSpriteToDraw;

typedef struct ChildScreenSpriteToDraw {
	uint32 image;
	int16 x,y;
	struct ChildScreenSpriteToDraw *next;
} ChildScreenSpriteToDraw;

typedef struct ParentSpriteToDraw {
	uint32 image;
	int16 left, top, right, bottom;
	int16 tile_x, tile_y;
	int16 tile_right, tile_bottom;
	ChildScreenSpriteToDraw *child;
	byte unk16;
	byte tile_z;
	byte tile_z_bottom;
} ParentSpriteToDraw;

typedef struct ViewportDrawer {
	DrawPixelInfo dpi;
	
	byte *spritelist_mem, *eof_spritelist_mem;
	
	StringSpriteToDraw **last_string, *first_string;
	TileSpriteToDraw **last_tile, *first_tile;

	ChildScreenSpriteToDraw **last_child;

	ParentSpriteToDraw **parent_list;

	byte combine_sprites;

	int offs_x, offs_y; // used when drawing ground sprites relative
	bool ground_child;
} ViewportDrawer;

static ViewportDrawer *_cur_vd;

TileHighlightData * const _thd_ptr = &_thd;
static TileInfo *_cur_ti;


Point MapXYZToViewport(ViewPort *vp, uint x, uint y, uint z)
{
	Point p;

	p.x = (y-x)*2 - (vp->virtual_width>>1);
	p.y = y+x-z - (vp->virtual_height>>1);
	return p;
}

void AssignWindowViewport(Window *w, int x, int y, 
	int width, int height, uint32 follow_flags, byte zoom)
{
	ViewPort *vp;
	Point pt;
	byte z;

	for(vp=_viewports; ; vp++) {
		assert(vp!=endof(_viewports));
		if (vp->width == 0)
			break;
	}

	vp->left = x + w->left;
	vp->top = y + w->top;
	vp->width = width;
	vp->height = height;
	
	vp->zoom = zoom;

	vp->virtual_width = width << zoom;
	vp->virtual_height = height << zoom;

	if (follow_flags & 0x80000000) {
		Vehicle *veh;
		WP(w,vp_d).follow_vehicle = (VehicleID)(follow_flags & 0xFFFF);
		veh = &_vehicles[WP(w,vp_d).follow_vehicle];
		pt = MapXYZToViewport(vp, veh->x_pos, veh->y_pos, veh->z_pos);
	} else {
		int x = GET_TILE_X(follow_flags) * 16;
		int y = GET_TILE_Y(follow_flags) * 16;
		WP(w,vp_d).follow_vehicle = 0xFFFF;
		z = (byte)GetSlopeZ(x,y);
		pt = MapXYZToViewport(vp, x,y, z);
	}

	WP(w,vp_d).scrollpos_x = pt.x;
	WP(w,vp_d).scrollpos_y = pt.y;
	w->viewport = vp;
	vp->virtual_left = 0;//pt.x;
	vp->virtual_top = 0;//pt.y;
}

static Point _vp_move_offs;

static void DoSetViewportPosition(Window *w, int left, int top, int width, int height)
{
	for (; w < _last_window; w++) {
		if (left + width > w->left &&
				w->left+w->width > left &&
				top + height > w->top &&
				w->top+w->height > top) {
		
			if (left < w->left) {
				DoSetViewportPosition(w, left, top, w->left - left, height);
				DoSetViewportPosition(w, left + (w->left - left), top, width - (w->left - left), height);
				return;
			}

			if (left + width > w->left + w->width) {
				DoSetViewportPosition(w, left, top, (w->left + w->width - left), height);
				DoSetViewportPosition(w, left + (w->left + w->width - left), top, width - (w->left + w->width - left) , height);
				return;
			}

			if (top < w->top) {
				DoSetViewportPosition(w, left, top, width, (w->top - top));
				DoSetViewportPosition(w, left, top + (w->top - top), width, height - (w->top - top));
				return;
			}

			if (top + height > w->top + w->height) {
				DoSetViewportPosition(w, left, top, width, (w->top + w->height - top));
				DoSetViewportPosition(w, left, top + (w->top + w->height - top), width , height - (w->top + w->height - top));
				return;
			}

			return;
		}
	}

	{
		int xo = _vp_move_offs.x;
		int yo = _vp_move_offs.y;

		if (abs(xo) >= width || abs(yo) >= height) {
			/* fully_outside */
			RedrawScreenRect(left, top, left+width, top+height);
			return;
		}

		GfxScroll(left, top, width, height, xo, yo);

		if (xo > 0) {
			RedrawScreenRect(left, top, xo + left, top + height);
			left += xo;
			width -= xo;
		} else if (xo < 0) {
			RedrawScreenRect(left+width+xo, top, left+width, top + height);
			width += xo;
		}

		if (yo > 0) {
			RedrawScreenRect(left, top, width+left, top + yo);
		} else if (yo < 0) {
			RedrawScreenRect(left, top + height + yo, width+left, top + height);
		}
	}
}

void SetViewportPosition(Window *w, int x, int y)
{
	ViewPort *vp = w->viewport;
	int old_left = vp->virtual_left;
	int old_top = vp->virtual_top;
	int i;
	int left, top, width, height;

	vp->virtual_left = x;
	vp->virtual_top = y;

	old_left >>= vp->zoom;
	old_top >>= vp->zoom;
	x >>= vp->zoom;
	y >>= vp->zoom;

	old_left -= x;
	old_top -= y;

	if (old_top == 0 && old_left == 0)
		return;

	_vp_move_offs.x = old_left;
	_vp_move_offs.y = old_top;

	left = vp->left;
	top = vp->top;
	width = vp->width;
	height = vp->height;

	if (left < 0) {
		width += left;
		left = 0;
	}

	if ( (i=(left + width - _screen.width)) >= 0)
		width -= i;

	if (width > 0) {
		if (top < 0) {
			height += top;
			top = 0;
		}
			
		if ( (i=(top + height - _screen.height)) >= 0) {
			height -= i;
		}

		if (height > 0)
			DoSetViewportPosition(w+1, left, top, width, height);
	}
}


ViewPort *IsPtInWindowViewport(Window *w, int x, int y)
{
	ViewPort *vp = w->viewport;
	if (vp != NULL &&
	    IS_INT_INSIDE(x, vp->left, vp->left + vp->width) &&
			IS_INT_INSIDE(y, vp->top, vp->top + vp->height))
		return vp;

	return NULL;
}

Point TranslateXYToTileCoord(ViewPort *vp, int x, int y) {
	int z;
	Point pt;
	int a,b;

	if ( (uint)(x -= vp->left) >= (uint)vp->width ||
				(uint)(y -= vp->top) >= (uint)vp->height) {
				Point pt = {-1, -1};
				return pt;
	}

	x <<= vp->zoom;
	y <<= vp->zoom;

	x += vp->virtual_left;
	y += vp->virtual_top;

	x >>= 2;
	y >>= 1;

	a = y-x;
	b = y+x;

	z = (byte)GetSlopeZ(a, b) >> 1;
	z = (byte)GetSlopeZ(a+z, b+z) >> 1;
	z = (byte)GetSlopeZ(a+z, b+z) >> 1;
	z = (byte)GetSlopeZ(a+z, b+z) >> 1;
	z = (byte)GetSlopeZ(a+z, b+z) >> 1;

	pt.x = a+z;
	pt.y = b+z;
	
	if ((uint)pt.x >= 0xFF0 || (uint)pt.y >= 0xFF0) {
		pt.x = pt.y = -1;
	}

	return pt;
}

static Point GetTileFromScreenXY(int x, int y)
{	
	Window *w;
	ViewPort *vp;
	Point pt;
	
	if ( (w = FindWindowFromPt(x, y)) != NULL && 
			 (vp = IsPtInWindowViewport(w, x, y)) != NULL)
				return TranslateXYToTileCoord(vp, x, y);

	pt.y = pt.x = -1;
	return pt;
}

Point GetTileBelowCursor()
{
	return GetTileFromScreenXY(_cursor.pos.x, _cursor.pos.y);
}

Point GetTileZoomCenter(bool in)
{
	int x, y;
	
	if (in)	{
		x = (_cursor.pos.x >> 1) + (_screen.width >> 2);
		y = (_cursor.pos.y >> 1) + (_screen.height >> 2);
	}
	else {
		x = _screen.width - _cursor.pos.x;
		y = _screen.height - _cursor.pos.y;
	}
	return GetTileFromScreenXY(x, y);
}

void DrawGroundSpriteAt(uint32 image, int x, int y, byte z)
{
	ViewportDrawer *vd = _cur_vd;
	TileSpriteToDraw *ts;
	
	_added_tile_sprite = true;

	assert( (image & 0x3fff) < NUM_SPRITES);

	ts = (TileSpriteToDraw*)vd->spritelist_mem;
	if ((byte*)ts >= vd->eof_spritelist_mem) {
		printf("Out of sprite mem\n");
		return;
	}

	vd->spritelist_mem += sizeof(TileSpriteToDraw);

	ts->image = image;
	ts->next = NULL;
	ts->x = x;
	ts->y = y;
	ts->z = z;
	*vd->last_tile = ts;
	vd->last_tile = &ts->next;	
}

void DrawGroundSprite(uint32 image)
{
	if (_cur_vd->ground_child) {
		AddChildSpriteScreen(image, _cur_vd->offs_x, _cur_vd->offs_y);
	} else {
		DrawGroundSpriteAt(image, _cur_ti->x, _cur_ti->y, _cur_ti->z);
	}
}


void OffsetGroundSprite(int x, int y)
{
	_cur_vd->offs_x = x;
	_cur_vd->offs_y = y;
	_cur_vd->ground_child = true;
}

void OffsetGroundSpriteEnd()
{
	_cur_vd->ground_child = false;
}


static void AddCombinedSprite(uint32 image, int x, int y, byte z)
{
	ViewportDrawer *vd = _cur_vd;
	int t;
	uint32 image_org = image;
	const SpriteDimension *sd;

	REMAP_COORDS(x,y,z);
	
	sd = GetSpriteDimension(image & 0x3FFF);

	if ((t = x + sd->xoffs) >= vd->dpi.left + vd->dpi.width || 
			(t + sd->xsize) <= vd->dpi.left ||
			(t = y + sd->yoffs) >= vd->dpi.top + vd->dpi.height ||
			(t + sd->ysize) <= vd->dpi.top)
		return;

	AddChildSpriteScreen(image_org, x - vd->parent_list[-1]->left, y - vd->parent_list[-1]->top);
}


void AddSortableSpriteToDraw(uint32 image, int x, int y, int w, int h, byte dz, byte z)
{
	ViewportDrawer *vd = _cur_vd;
	ParentSpriteToDraw *ps;
	const SpriteDimension *sd;

	assert( (image & 0x3fff) < NUM_SPRITES);

	if (vd->combine_sprites == 2) {
		AddCombinedSprite(image, x, y, z);
		return;
	}

	vd->last_child = NULL;

	ps = (ParentSpriteToDraw*)vd->spritelist_mem;
	if ((byte*)ps >= vd->eof_spritelist_mem) {
		printf("Out of sprite mem\n");
		return;
	}

	ps->image = image;
	ps->tile_x = x;
	ps->tile_right = x + w - 1;

	ps->tile_y = y;
	ps->tile_bottom = y + h - 1;

	ps->tile_z = z;
	ps->tile_z_bottom = z + dz - 1;

	REMAP_COORDS(x,y,z);

	sd = GetSpriteDimension(image & 0x3FFF);
	if ((ps->left = (x += sd->xoffs)) >= vd->dpi.left + vd->dpi.width || 
			(ps->right = (x + sd->xsize)) <= vd->dpi.left ||
			(ps->top = (y += sd->yoffs)) >= vd->dpi.top + vd->dpi.height ||
			(ps->bottom = (y + sd->ysize)) <= vd->dpi.top) {
		return;
	}

	ps->unk16 = 0;
	ps->child = NULL;
	vd->last_child = &ps->child;

	vd->spritelist_mem += sizeof(ParentSpriteToDraw);
	*vd->parent_list++ = ps;

	if (vd->combine_sprites == 1) {
		vd->combine_sprites = 2;
	}
}

void StartSpriteCombine()
{
	_cur_vd->combine_sprites = 1;
}

void EndSpriteCombine()
{
	_cur_vd->combine_sprites = 0;
}

void AddChildSpriteScreen(uint32 image, int x, int y)
{
	ViewportDrawer *vd = _cur_vd;
	ChildScreenSpriteToDraw *cs;

	cs = (ChildScreenSpriteToDraw*) vd->spritelist_mem;
	if ((byte*)cs >= vd->eof_spritelist_mem) {
		printf("Out of sprite mem\n");
		return;
	}

	if (vd->last_child == NULL)
		return;

	*vd->last_child = cs;
	vd->last_child = &cs->next;

	vd->spritelist_mem += sizeof(ChildScreenSpriteToDraw);

	cs->image = image;
	cs->x = x;
	cs->y = y;
	cs->next = NULL;
}

/* Returns a StringSpriteToDraw */
void *AddStringToDraw(int x, int y, StringID string, uint32 params_1, uint32 params_2)
{
	ViewportDrawer *vd = _cur_vd;
	StringSpriteToDraw *ss;

	ss = (StringSpriteToDraw*)vd->spritelist_mem;
	if ((byte*)ss >= vd->eof_spritelist_mem) {
		printf("Out of sprite mem\n");
		return NULL;
	}

	vd->spritelist_mem += sizeof(StringSpriteToDraw);

	ss->string = string;
	ss->next = NULL;
	ss->x = x;
	ss->y = y;
	ss->params[0] = params_1;
	ss->params[1] = params_2;
	ss->width = 0;

	*vd->last_string = ss;
	vd->last_string = &ss->next;

	return ss;
}

/* Debugging code */

#ifdef DEBUG_TILE_PUSH
static int _num_push;
static uint _pushed_tile[200];
static int _pushed_track[200];

static uint _stored_tile[200];
static int _stored_track[200];
static int _num_stored;

void dbg_store_path()
{
	memcpy(_stored_tile, _pushed_tile, sizeof(_stored_tile));
	memcpy(_stored_track, _pushed_track, sizeof(_stored_tile));
	_num_stored = _num_push;
	MarkWholeScreenDirty();
}

void dbg_push_tile(uint tile, int track)
{
	_pushed_tile[_num_push] = tile;
	_pushed_track[_num_push++] = track;
	dbg_store_path();
}

void dbg_pop_tile()
{
	_num_push--;
}

static const uint16 _dbg_track_sprite[] = {
	0x3F4,
	0x3F3,
	0x3F5,
	0x3F6,
	0x3F8,
	0x3F7,
};

int dbg_draw_pushed(const TileInfo *ti)
{
	int i;
	if (ti->tile==0)
		return 0;
	for(i=0; i!=_num_stored; i++)
		if (_stored_tile[i] == ti->tile) {
			DrawGroundSpriteAt(_dbg_track_sprite[_stored_track[i]&7], ti->x, ti->y, ti->z);
		}
	return -1;
}

#endif

static void DrawSelectionSprite(uint32 image, const TileInfo *ti)
{
	if (_added_tile_sprite) {
		DrawGroundSpriteAt(image, ti->x, ti->y, ti->z + 7);
	} else {
		AddSortableSpriteToDraw(image, ti->x, ti->y, 0x10, 0x10, 1, ti->z + 7);
	}
}

static void DrawTileSelection(const TileInfo *ti)
{
	uint32 image;
	const TileHighlightData *thd = _thd_ptr;

#ifdef DEBUG_TILE_PUSH
	dbg_draw_pushed(ti);
#endif
	
	// Draw a red error square?
	if (ti->tile != 0 && thd->redsq == (TileIndex)ti->tile) {
		DrawSelectionSprite(0x030382F0 | _tileh_to_sprite[ti->tileh], ti);
		return;
	}

	if (!(thd->flags & (THD_RECT | THD_DOT)))
		return;

	// Check if it's inside the inner area
	if (IS_INSIDE_1D(ti->x, thd->pos.x, thd->size.x) &&
			IS_INSIDE_1D(ti->y, thd->pos.y, thd->size.y)) {
		
		// Draw either a rect or a single dot.
		if (!(thd->flags & THD_DOT)) {
			image = 0x2F0 + _tileh_to_sprite[ti->tileh];
			if (thd->make_square_red) image |= 0x3048000;
			DrawSelectionSprite(image, ti);
		} else {
			// Figure out the Z coordinate for the single dot.
			byte z = ti->z;
			if (ti->tileh & 8) {
				z += 8;				
				if (!(ti->tileh & 2) && (ti->tileh & 0x10)) {	
					z += 8;
				}
			}
			DrawGroundSpriteAt(_cur_dpi->zoom != 2 ? 0x306 : 0xFEE,ti->x, ti->y, z);
		}
		return;
	}

	// Check if it's inside the outer area.
	if (thd->size.x < thd->maxsize.x &&
			IS_INSIDE_1D(ti->x, thd->pos.x + thd->offs.x, thd->maxsize.x) &&
			IS_INSIDE_1D(ti->y, thd->pos.y + thd->offs.y, thd->maxsize.y)) {
		// Draw a blue rect.
		DrawSelectionSprite(0x30582F0 | _tileh_to_sprite[ti->tileh], ti);
		return;
	}
}

void ViewportAddLandscape()
{
	ViewportDrawer *vd = _cur_vd;
	int x, y, width, height;
	TileInfo ti;
	bool direction;

	_cur_ti = &ti;

	// Transform into tile coordinates
	x = (vd->dpi.top >> 1) - (vd->dpi.left >> 2);
	y = (vd->dpi.top >> 1) + (vd->dpi.left >> 2) - 0x10;

	// and round to closest full tile
	x &= ~0xF;
	y &= ~0xF;

	/* determine size of area */
	width = (vd->dpi.left + vd->dpi.width-((y - x) << 1) + 95) >> 6;
	height = (vd->dpi.top + vd->dpi.height - (x + y) + 241) >> 5 << 1;

	assert(width > 0);
	assert(height > 0);
	
	direction = false;

	do {
		int width_cur = width;
		int x_cur = x;
		int y_cur = y;
		
		do {
			FindLandscapeHeight(&ti, x_cur, y_cur);
			y_cur += 0x10;
			x_cur -= 0x10;

			_added_tile_sprite = false;
			
			DrawTile(&ti);
			DrawTileSelection(&ti);
		} while (--width_cur);
	
		if ( (direction^=1) != 0)
			y += 0x10;
		else
			x += 0x10;

	} while (--height);	
}


void ViewportAddCityNames(DrawPixelInfo *dpi)
{
	City *ci;
	int left, top, right, bottom;

	if (!(_display_opt & DO_SHOW_TOWN_NAMES) || _game_mode == GM_MENU)
		return;

	left = dpi->left;
	top = dpi->top;
	right = left + dpi->width;
	bottom = top + dpi->height;

	if (dpi->zoom < 1) {
		FOR_ALL_CITIES(ci) {
			if (ci->xy &&
			    bottom > ci->sign_top &&
					top < ci->sign_top + 12 &&
					right > ci->sign_left &&
					left < ci->sign_left + ci->namewidth_1) {
				
				AddStringToDraw(ci->sign_left + 1, ci->sign_top + 1, STR_2001, ci->citynametype, ci->citynameparts);
			}
		}
	} else if (dpi->zoom == 1) {
		right += 2;
		bottom += 2;

		FOR_ALL_CITIES(ci) {
			if (ci->xy &&
			    bottom > ci->sign_top &&
					top < ci->sign_top + 24 &&
					right > ci->sign_left &&
					left < ci->sign_left + ci->namewidth_1*2) {
				
				AddStringToDraw(ci->sign_left + 1, ci->sign_top + 1, STR_2001, ci->citynametype, ci->citynameparts);
			}	
		}
	} else {
		right += 4;
		bottom += 5;

		assert(dpi->zoom == 2);
		FOR_ALL_CITIES(ci) {
			if (ci->xy &&
			    bottom > ci->sign_top &&
					top < ci->sign_top + 24 &&
					right > ci->sign_left &&
					left < ci->sign_left + ci->namewidth_2*4) {
				
				AddStringToDraw(ci->sign_left + 5, ci->sign_top + 1, STR_2002, ci->citynametype, ci->citynameparts);
				AddStringToDraw(ci->sign_left + 1, ci->sign_top - 3, STR_2003, ci->citynametype, ci->citynameparts);
			}				
		}
	}
}

void ViewportAddStationNames(DrawPixelInfo *dpi)
{
	int left, top, right, bottom;
	Station *st;
	StringSpriteToDraw *sstd;

	if (!(_display_opt & DO_SHOW_STATION_NAMES) || _game_mode == GM_MENU)
		return;

	left = dpi->left;
	top = dpi->top;
	right = left + dpi->width;
	bottom = top + dpi->height;

	if (dpi->zoom < 1) {
		FOR_ALL_STATIONS(st) {
			if (st->xy &&
			    bottom > st->sign_top &&
					top < st->sign_top + 12 &&
					right > st->sign_left &&
					left < st->sign_left + st->namewidth_1) {
				
				sstd=AddStringToDraw(st->sign_left + 1, st->sign_top + 1, STR_305C_0, st->index, st->facilities);
				if (sstd != NULL) {
					sstd->color = (st->owner == 0x10 || (st->had_vehicle_of_type & HVOT_PENDING_DELETE)) ? 0xE : _player_colors[st->owner];
					sstd->width = st->namewidth_1;
				}
			}
		}
	} else if (dpi->zoom == 1) {
		right += 2;
		bottom += 2;

		FOR_ALL_STATIONS(st) {
			if (st->xy &&
			    bottom > st->sign_top &&
					top < st->sign_top + 24 &&
					right > st->sign_left &&
					left < st->sign_left + st->namewidth_1*2) {
				
				sstd=AddStringToDraw(st->sign_left + 1, st->sign_top + 1, STR_305C_0, st->index, st->facilities);
				if (sstd != NULL) {
					sstd->color = (st->owner == 0x10 || (st->had_vehicle_of_type & HVOT_PENDING_DELETE)) ? 0xE : _player_colors[st->owner];
					sstd->width = st->namewidth_1;
				}
			}
		}

	} else {
		assert(dpi->zoom == 2);

		right += 4;
		bottom += 5;

		FOR_ALL_STATIONS(st) {
			if (st->xy &&
			    bottom > st->sign_top &&
					top < st->sign_top + 24 &&
					right > st->sign_left &&
					left < st->sign_left + st->namewidth_2*4) {
				
				sstd=AddStringToDraw(st->sign_left + 1, st->sign_top + 1, STR_305D_0, st->index, st->facilities);
				if (sstd != NULL) {
					sstd->color = (st->owner == 0x10 || (st->had_vehicle_of_type & HVOT_PENDING_DELETE)) ? 0xE : _player_colors[st->owner];
					sstd->width = st->namewidth_2 | 0x8000;
				}
			}
		}
	}
}

void ViewportAddSigns(DrawPixelInfo *dpi)
{
	SignStruct *ss;
	int left, top, right, bottom;
	StringSpriteToDraw *sstd;

	if (!(_display_opt & DO_SHOW_SIGNS))
		return;

	left = dpi->left;
	top = dpi->top;
	right = left + dpi->width;
	bottom = top + dpi->height;

	if (dpi->zoom < 1) {
		for(ss=_sign_list; ss != endof(_sign_list); ss++) {
			if (ss->str &&
					bottom > ss->sign_top &&
					top < ss->sign_top + 12 &&
					right > ss->sign_left &&
					left < ss->sign_left + ss->namewidth_1) {
				
				sstd=AddStringToDraw(ss->sign_left + 1, ss->sign_top + 1, STR_2806, ss->str, 0);
				if (sstd != NULL) {
					sstd->width = ss->namewidth_1;
					sstd->color = 14;
				}
			}
		}	
	} else if (dpi->zoom == 1) {
		right += 2;
		bottom += 2;
		for(ss=_sign_list; ss != endof(_sign_list); ss++) {
			if (ss->str &&
					bottom > ss->sign_top &&
					top < ss->sign_top + 24 &&
					right > ss->sign_left &&
					left < ss->sign_left + ss->namewidth_1*2) {
				
				sstd=AddStringToDraw(ss->sign_left + 1, ss->sign_top + 1, STR_2806, ss->str, 0);
				if (sstd != NULL) {
					sstd->width = ss->namewidth_1;
					sstd->color = 14;
				}
			}
		}	
	} else {
		right += 4;
		bottom += 5;

		for(ss=_sign_list; ss != endof(_sign_list); ss++) {
			if (ss->str &&
					bottom > ss->sign_top &&
					top < ss->sign_top + 24 &&
					right > ss->sign_left &&
					left < ss->sign_left + ss->namewidth_2*4) {
				
				sstd=AddStringToDraw(ss->sign_left + 1, ss->sign_top + 1, STR_2807, ss->str, 0);
				if (sstd != NULL) {
					sstd->width = ss->namewidth_2;
					sstd->color = 14;
				}
			}
		}	
	}
}

void ViewportDrawTileSprites(TileSpriteToDraw *ts)
{	
	do {
		DrawSprite(ts->image, (ts->y - ts->x)*2, ts->x + ts->y - ts->z);
	} while ( (ts = ts->next) != NULL);
}

#define NEW_SORT_METHOD

#if !defined(NEW_SORT_METHOD)
static const byte _sort_sprites_table[64] = {
	1, 0, 0, 0, 0, 0, 0, 0,
	1, 0, 0, 0, 0, 0, 0, 0,
	1, 0, 0, 0, 0, 0, 0, 0,
	1, 0, 0, 0, 0, 0, 0, 0,
	1, 0, 0, 0, 0, 0, 0, 0,
	1, 0, 0, 0, 0, 0, 0, 0,
	1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
};
#endif

void ViewportSortParentSprites(ParentSpriteToDraw **psd)
{
	ParentSpriteToDraw *ps, *ps2,*ps3, **psd2, **psd3;
#if !defined(NEW_SORT_METHOD)
	int cond;
#endif

	while((ps=*psd) != NULL) {
		if (!(ps->unk16 & 1)) {
			ps->unk16 |= 1;
			psd2 = psd;
			
			while ( (ps2=*++psd2) != NULL) {
				if (ps2->unk16 & 1)
					continue;

#if !defined(NEW_SORT_METHOD)
				cond = ((ps->tile_x < ps2->tile_right) ? 32 : 0) +
					((ps->tile_y < ps2->tile_bottom) ? 16 : 0) + 
					((ps->tile_z < ps2->tile_z_bottom) ? 8 : 0) + 
					((ps->tile_right < ps2->tile_x) ? 4 : 0) + 
					((ps->tile_bottom < ps2->tile_y) ? 2 : 0) + 
					((ps->tile_z_bottom < ps2->tile_z) ? 1 : 0);

				if (_sort_sprites_table[cond]) {
#else
				if (ps->tile_right >= ps2->tile_x &&
						ps->tile_bottom >= ps2->tile_y &&
						ps->tile_z_bottom >= ps2->tile_z && (
						ps->tile_x >= ps2->tile_right ||
						ps->tile_y >= ps2->tile_bottom ||
						ps->tile_z >= ps2->tile_z_bottom
						)) {
#endif
					psd3 = psd;
					do {
						ps3 = *psd3;
						*psd3 = ps2;
						ps2 = ps3;

						psd3++;
					} while (psd3 <= psd2);
				}
			}
		} else {
			psd++;
		}
	}
}

void ViewportDrawParentSprites(ParentSpriteToDraw **psd)
{
	ParentSpriteToDraw *ps;
	ChildScreenSpriteToDraw *cs;
	int x,y;

	for (;(ps=*psd) != NULL;psd++) {
		x = ps->tile_x;
		y = ps->tile_y;
		REMAP_COORDS(x,y,ps->tile_z);
		DrawSprite(ps->image, x, y);

		cs = ps->child;
		while (cs) {
			DrawSprite(cs->image, ps->left + cs->x, ps->top + cs->y);
			cs = cs->next;
		}
	}
}

void ViewportDrawStrings(DrawPixelInfo *dpi, StringSpriteToDraw *ss)
{
	DrawPixelInfo dp;
	byte zoom;
	
	_cur_dpi = &dp;
	dp = *dpi;

	zoom = (byte)dp.zoom;
	dp.zoom = 0;

	dp.left >>= zoom;
	dp.top >>= zoom;
	dp.width >>= zoom;
	dp.height >>= zoom;

	do {

		if (ss->width != 0) {
			int x, y, w, bottom;
			
			x = (ss->x >> zoom) - 1;
			y = (ss->y >> zoom) - 1;

			bottom = y + 11;

			w = ss->width;
			if (w & 0x8000) {
				w &= ~0x8000;
				y--;
				bottom -= 6;
				w -= 3;
			}

			DrawFrameRect(x,y, x+w, bottom, ss->color, 0);
		}

		SET_DPARAM32(0, ss->params[0]);
		SET_DPARAM32(1, ss->params[1]);

		DrawString(ss->x >> zoom, (ss->y >> zoom) - (ss->width&0x8000?2:0), ss->string, 0);
	} while ( (ss = ss->next) != NULL);

	_cur_dpi = dpi;
}

void ViewportDoDraw(ViewPort *vp, int left, int top, int right, int bottom)
{
	ViewportDrawer vd;
	int mask;
	int x,y;
	DrawPixelInfo *old_dpi;

	byte mem[VIEWPORT_DRAW_MEM];
	ParentSpriteToDraw *parent_list[750];

	_cur_vd = &vd;

	old_dpi = _cur_dpi;
	_cur_dpi = &vd.dpi;

	vd.dpi.zoom = vp->zoom;
	mask = (-1) << vp->zoom;

	vd.combine_sprites = 0;
	vd.ground_child = 0;

	vd.dpi.width = (right - left) & mask;
	vd.dpi.height = (bottom - top) & mask;
	vd.dpi.left = left & mask;
	vd.dpi.top = top & mask;
	vd.dpi.pitch = old_dpi->pitch;

	x = ((vd.dpi.left - (vp->virtual_left&mask)) >> vp->zoom) + vp->left;
	y = ((vd.dpi.top - (vp->virtual_top&mask)) >> vp->zoom) + vp->top;

	vd.dpi.dst_ptr = old_dpi->dst_ptr + x - old_dpi->left + (y - old_dpi->top) * old_dpi->pitch;

	vd.parent_list = parent_list;
	vd.spritelist_mem = mem;
	vd.eof_spritelist_mem = &mem[sizeof(mem) - 0x40];
	vd.last_string = &vd.first_string;
	vd.first_string = NULL;
	vd.last_tile = &vd.first_tile;
	vd.first_tile = NULL;
 
	ViewportAddLandscape();
	ViewportAddVehicles(&vd.dpi);
	DrawTextEffects(&vd.dpi);

	ViewportAddCityNames(&vd.dpi);
	ViewportAddStationNames(&vd.dpi);
	ViewportAddSigns(&vd.dpi);

	if (vd.first_tile != NULL)
		ViewportDrawTileSprites(vd.first_tile);

	assert(vd.parent_list - parent_list <= lengthof(parent_list));

	/* null terminate parent sprite list */
	*vd.parent_list = NULL;

	ViewportSortParentSprites(parent_list);
	ViewportDrawParentSprites(parent_list);
	
	if (vd.first_string != NULL)
		ViewportDrawStrings(&vd.dpi, vd.first_string);

	_cur_dpi = old_dpi;
}

// Make sure we don't draw a too big area at a time.
// If we do, the sprite memory will overflow.
static void ViewportDrawChk(ViewPort *vp, int left, int top, int right, int bottom)
{
	if (((bottom - top) * (right - left) << vp->zoom) > 180000) {
		if ((bottom - top) > (right - left)) {
			int t = (top + bottom) >> 1;
			ViewportDrawChk(vp, left, top, right, t);
			ViewportDrawChk(vp, left, t, right, bottom);
		} else {
			int t = (left + right) >> 1;
			ViewportDrawChk(vp, left, top, t, bottom);
			ViewportDrawChk(vp, t, top, right, bottom);
		}
	} else {
		ViewportDoDraw(vp, 
			((left - vp->left) << vp->zoom) + vp->virtual_left,
			((top - vp->top) << vp->zoom) + vp->virtual_top,
			((right - vp->left) << vp->zoom) + vp->virtual_left,
			((bottom - vp->top) << vp->zoom) + vp->virtual_top
		);
	}
}

static void INLINE ViewportDraw(ViewPort *vp, int left, int top, int right, int bottom)
{
	int t;

	if (right <= vp->left ||
			bottom <= vp->top)
				return;

	if (left >= (t=vp->left + vp->width))
		return;

	if (left < vp->left) left = vp->left;
	if (right > t) right = t;

	if (top >= (t = vp->top + vp->height))
		return;

	if (top < vp->top) top = vp->top;
	if (bottom > t) bottom = t;

	ViewportDrawChk(vp, left, top, right, bottom);
}

void DrawWindowViewport(Window *w) {
	DrawPixelInfo *dpi = _cur_dpi;

	dpi->left += w->left;
	dpi->top += w->top;

	ViewportDraw(w->viewport, dpi->left, dpi->top, dpi->left + dpi->width, dpi->top + dpi->height);

	dpi->left -= w->left;
	dpi->top -= w->top;

}

void UpdateViewportPosition(Window *w)
{
	ViewPort *vp = w->viewport;

	if (WP(w,vp_d).follow_vehicle != 0xFFFF) {
		Vehicle *veh;
		Point pt;

		veh = &_vehicles[WP(w,vp_d).follow_vehicle];
		pt = MapXYZToViewport(vp, veh->x_pos, veh->y_pos, veh->z_pos);
		SetViewportPosition(w, pt.x, pt.y);
	} else {
		int x,y,t;
		int err;

		x = WP(w,vp_d).scrollpos_x >> 2;
		y = WP(w,vp_d).scrollpos_y >> 1;

		t = x;
		x = y - t;
		y = y + t;

		t = (-130) << vp->zoom;
		err = 0;
		if (y < t || y > (t += 0xFEF)) {
			y = t;
			err++;
		}

		if (x < (t=0) || x > (t=0xFEF) ) {
			x = t;
			err++;
		}

		if (err != 0) {
			/* coordinate remap */
			REMAP_COORDS(x, y, 0);
		
			t = (-1) << vp->zoom; 
			
			WP(w,vp_d).scrollpos_x = x & t;
			WP(w,vp_d).scrollpos_y = y & t;
		}
		SetViewportPosition(w, WP(w,vp_d).scrollpos_x, WP(w,vp_d).scrollpos_y);
	}
}

void SetObjectToPlaceWnd(uint32 icon, byte mode, Window *w)
{
	SetObjectToPlace(icon,mode,w->window_class, w->window_number);
}

void SetObjectToPlace(uint32 icon, byte mode, byte window_class, uint16 window_num)
{
	TileHighlightData *thd = _thd_ptr;
	byte curs;
	Window *w;

	if (thd->place_mode != 0) {
		thd->place_mode = 0;
		w = FindWindowById(thd->window_class, thd->window_number);
		if (w != NULL)
			CallWindowEventNP(w, WE_ABORT_PLACE_OBJ);
	}
	
	SetTileSelectSize(1, 1);
	
	thd->make_square_red = false;

	curs = mode;
	mode &= 0xF;

	if (mode == 4) {
		mode = 0;
		_special_mouse_mode = WSM_DRAGDROP;
	} else {
		_special_mouse_mode = WSM_NONE;
	}

	thd->place_mode = mode;
	thd->window_class = window_class;
	thd->window_number = window_num;

	if (curs & 0x10)
		SetAnimatedMouseCursor((uint16*)icon);
	else
		SetMouseCursor(icon);
}

void ResetObjectToPlace(){
	SetObjectToPlace(0,0,0,0);
}

void MarkViewportDirty(ViewPort *vp, int left, int top, int right, int bottom)
{
	if ( (right -= vp->virtual_left) <= 0)
		return;

	if ( (bottom -= vp->virtual_top) <= 0)
		return;

	if ( (left -= vp->virtual_left) < 0)
		left = 0;

	if ((uint)left >= (uint)vp->virtual_width)
		return;

	if ( (top -= vp->virtual_top) < 0)
		top = 0;

	if ((uint)top >= (uint)vp->virtual_height)
		return;

	SetDirtyBlocks(
		(left >> vp->zoom) + vp->left,
		(top >> vp->zoom) + vp->top,
		(right >> vp->zoom) + vp->left,
		(bottom >> vp->zoom) + vp->top
	);
}

void MarkAllViewportsDirty(int left, int top, int right, int bottom)
{
	int count = 8;
	ViewPort *vp = _viewports;
	do {
		if (vp->width != 0)
			MarkViewportDirty(vp, left, top, right, bottom);
	} while (vp++,--count);
}

void MarkTileDirtyByTile(TileIndex tile) {
	int x,y,z;

	z = GetTileZ(tile);

	x = GET_TILE_X(tile) << 4;
	y = GET_TILE_Y(tile) << 4;

	REMAP_COORDS(x, y, z);

	MarkAllViewportsDirty(
		x - 31,
		y - 122,
		x - 31 + 67,
		y - 122 + 154
	);
}

void MarkTileDirty(int x, int y)
{
	int z = 0;
	if (IS_INT_INSIDE(x, 0, TILES_X*16) &&
			IS_INT_INSIDE(y, 0, TILES_X*16))
		z = GetTileZ(TILE_FROM_XY(x,y));
	
	REMAP_COORDS(x, y, z);

	MarkAllViewportsDirty(
		x - 31,
		y - 122,
		x - 31 + 67,
		y - 122 + 154
	);
}	

void SetSelectionTilesDirty(int x, int y)
{
	int y_size, x_size;
	TileHighlightData *thd = _thd_ptr;

	if ((x_size=thd->size.x) < thd->maxsize.x) {
		x_size = thd->maxsize.x;
		x += thd->offs.x;
	}

	if ((y_size=thd->size.y) < thd->maxsize.y) {
		y_size = thd->maxsize.y;
		y += thd->offs.y;
	}

	assert(x_size > 0);
	assert(y_size > 0);

	x_size += x;
	y_size += y;

	do {
		int y_bk = y;
		do {
			MarkTileDirty(x, y);
		} while ( (y+=16) != y_size);
		y = y_bk;
	} while ( (x+=16) != x_size);
}


static bool CheckClickOnCity(ViewPort *vp, int x, int y)
{
	City *c;

	if (!(_display_opt & DO_SHOW_TOWN_NAMES))
		return false;

	if (vp->zoom < 1) {
		x = x - vp->left + vp->virtual_left;
		y = y - vp->top + vp->virtual_top;	
		
		FOR_ALL_CITIES(c) {
			if (c->xy &&
			    y >= c->sign_top &&
					y < c->sign_top + 12 &&
					x >= c->sign_left &&
					x < c->sign_left + c->namewidth_1) {
				ShowTownViewWindow(c->index);
				return true;
			}
		}
	} else if (vp->zoom == 1) {
		x = (x - vp->left + 1) * 2 + vp->virtual_left;
		y = (y - vp->top + 1) * 2 + vp->virtual_top;	
		FOR_ALL_CITIES(c) {
			if (c->xy &&
			    y >= c->sign_top &&
					y < c->sign_top + 24 &&
					x >= c->sign_left &&
					x < c->sign_left + c->namewidth_1 * 2) {
				ShowTownViewWindow(c->index);
				return true;
			}
		}
	} else {
		x = (x - vp->left + 3) * 4 + vp->virtual_left;
		y = (y - vp->top + 3) * 4 + vp->virtual_top;	
		FOR_ALL_CITIES(c) {
			if (c->xy &&
			    y >= c->sign_top &&
					y < c->sign_top + 24 &&
					x >= c->sign_left &&
					x < c->sign_left + c->namewidth_2 * 4) {
				ShowTownViewWindow(c->index);
				return true;
			}
		}
	}

	return false;
}

static bool CheckClickOnStation(ViewPort *vp, int x, int y)
{
	Station *st;

	if (!(_display_opt & DO_SHOW_STATION_NAMES))
		return false;

	if (vp->zoom < 1) {
		x = x - vp->left + vp->virtual_left;
		y = y - vp->top + vp->virtual_top;	

		FOR_ALL_STATIONS(st) {
			if (st->xy &&
			    y >= st->sign_top &&
					y < st->sign_top + 12 &&
					x >= st->sign_left &&
					x < st->sign_left + st->namewidth_1) {
				ShowStationViewWindow(st->index);
				return true;
			}
		}
	} else if (vp->zoom == 1) {
		x = (x - vp->left + 1) * 2 + vp->virtual_left;
		y = (y - vp->top + 1) * 2 + vp->virtual_top;	
		FOR_ALL_STATIONS(st) {
			if (st->xy &&
			    y >= st->sign_top &&
					y < st->sign_top + 24 &&
					x >= st->sign_left &&
					x < st->sign_left + st->namewidth_1 * 2) {
				ShowStationViewWindow(st->index);
				return true;
			}
		}
	} else {
		x = (x - vp->left + 3) * 4 + vp->virtual_left;
		y = (y - vp->top + 3) * 4 + vp->virtual_top;	
		FOR_ALL_STATIONS(st) {
			if (st->xy &&
			    y >= st->sign_top &&
					y < st->sign_top + 24 &&
					x >= st->sign_left &&
					x < st->sign_left + st->namewidth_2 * 4) {
				ShowStationViewWindow(st->index);
				return true;
			}
		}
	}

	return false;
}

static bool CheckClickOnSign(ViewPort *vp, int x, int y)
{
	SignStruct *ss;

	if (!(_display_opt & DO_SHOW_SIGNS))
		return false;

	if (vp->zoom < 1) {
		x = x - vp->left + vp->virtual_left;
		y = y - vp->top + vp->virtual_top;	

		for(ss = _sign_list; ss != endof(_sign_list); ss++) {
			if (ss->str &&
			    y >= ss->sign_top &&
					y < ss->sign_top + 12 &&
					x >= ss->sign_left &&
					x < ss->sign_left + ss->namewidth_1) {
				ShowRenameSignWindow(ss);
				return true;
			}
		}
	} else if (vp->zoom == 1) {
		x = (x - vp->left + 1) * 2 + vp->virtual_left;
		y = (y - vp->top + 1) * 2 + vp->virtual_top;	
		for(ss = _sign_list; ss != endof(_sign_list); ss++) {
			if (ss->str &&
			    y >= ss->sign_top &&
					y < ss->sign_top + 24 &&
					x >= ss->sign_left &&
					x < ss->sign_left + ss->namewidth_1 * 2) {
				ShowRenameSignWindow(ss);
				return true;
			}
		}
	} else {
		x = (x - vp->left + 3) * 4 + vp->virtual_left;
		y = (y - vp->top + 3) * 4 + vp->virtual_top;	
		for(ss = _sign_list; ss != endof(_sign_list); ss++) {
			if (ss->str &&
			    y >= ss->sign_top &&
					y < ss->sign_top + 24 &&
					x >= ss->sign_left &&
					x < ss->sign_left + ss->namewidth_2 * 4) {
				ShowRenameSignWindow(ss);
				return true;
			}
		}
	}

	return false;
}

static void CheckClickOnLandscape(ViewPort *vp, int x, int y)
{
	Point pt = TranslateXYToTileCoord(vp,x,y);
	if (pt.x != -1) {
		uint tile = TILE_FROM_XY(pt.x, pt.y);
		ClickTile(tile);
	}
}

void HandleViewportClicked(ViewPort *vp, int x, int y)
{
	if (CheckClickOnCity(vp, x, y))
		return;

	if (CheckClickOnStation(vp, x, y))
		return;

	if (CheckClickOnSign(vp, x, y))
		return;

	CheckClickOnLandscape(vp, x, y);
	CheckClickOnVehicle(vp, x, y);
}


void PlaceObject()
{
	Point pt;
	Window *w;
	WindowEvent e;
	
	pt = GetTileBelowCursor();
	if (pt.x == -1)
		return;

	_tile_fract_coords.x = pt.x & 0xF;
	_tile_fract_coords.y = pt.y & 0xF;

	if ((w = GetCallbackWnd()) != NULL) {
		e.event = WE_PLACE_OBJ;
		e.place.pt = pt;
		e.place.tile = TILE_FROM_XY(pt.x, pt.y);
		w->wndproc(w, &e);
	}
}

bool ScrollMainWindowTo(int x, int y)
{
	Window *w = FindWindowById(WC_MAIN_WINDOW, 0);
	Point pt;

	pt = MapXYZToViewport(w->viewport, x, y, (byte)GetSlopeZ(x, y));
	WP(w,vp_d).follow_vehicle = -1;

	if (WP(w,vp_d).scrollpos_x == pt.x &&
			WP(w,vp_d).scrollpos_y == pt.y)
				return false;

	WP(w,vp_d).scrollpos_x = pt.x;
	WP(w,vp_d).scrollpos_y = pt.y;
	return true;
}


bool ScrollMainWindowToTile(TileIndex tile)
{
	return ScrollMainWindowTo(GET_TILE_X(tile)*16, GET_TILE_Y(tile)*16);
}

void SetRedErrorSquare(TileIndex tile)
{
	TileIndex old;

	old = _thd.redsq;
	_thd.redsq = tile;

	if (tile != old) {
		if (tile != 0) MarkTileDirtyByTile(tile);
		if (old  != 0) MarkTileDirtyByTile(old);
	}
}


void SetTileSelectSize(int w, int h)
{
	TileHighlightData *thd = _thd_ptr;
	thd->to_place_size.x = thd->to_place_maxsize.x = w * 16;
	thd->to_place_size.y = thd->to_place_maxsize.y = h * 16;
}

void SetTileSelectBigSize(int ox, int oy, int sx, int sy) {
	TileHighlightData *thd = _thd_ptr;
	thd->offs.x = ox * 16;
	thd->offs.y = oy * 16;
	thd->to_place_maxsize.x += sx * 16;
	thd->to_place_maxsize.y += sy * 16;
}

void SetSelectionUnk1()
{
	TileHighlightData *thd = _thd_ptr;
	Point pt;
	int x1,y1;

	if (thd->place_mode == 0) {
default_mode:;
		if (thd->flags & (THD_RECT|THD_DOT)) {
			thd->flags &= ~(THD_RECT|THD_DOT);
			SetSelectionTilesDirty(thd->pos.x, thd->pos.y);
			thd->size = thd->to_place_size;
		}
		return;
	}
	
	if (thd->place_mode == 3) {
		int x2,y2;

		x1 = thd->select_from.x;
		if (x1 == -1) goto default_mode;
		y1 = thd->select_from.y;

		x2 = thd->select_to.x;
		y2 = thd->select_to.y;
		
		if (x1 >= x2) intswap(x1,x2);
		if (y1 >= y2) intswap(y1,y2);
		
		x2 = x2 - x1 + 16;
		y2 = y2 - y1 + 16;	

		thd->to_place_size.x = x2;
		thd->to_place_size.y = y2;

		if (thd->flags & THD_DOT) {	thd->flags ^= THD_DOT; goto thd3_stuff; }
		if (!(thd->flags & THD_RECT)) { thd->flags ^= THD_RECT; goto thd2_stuff; }

		if (thd->pos.x == x1 && thd->pos.y == y1 &&
				thd->size.x == x2 && thd->size.y == y2)
					return;
	} else {
		pt = GetTileBelowCursor();
		x1 = pt.x;
		y1 = pt.y;

		if (x1 == -1)
			goto default_mode;

		if (thd->place_mode == 1) {
			x1 &= ~0xF;
			y1 &= ~0xF;

			if (thd->flags & THD_DOT) { thd->flags ^= THD_DOT; goto thd3_stuff; }
			if (!(thd->flags & THD_RECT)) { thd->flags ^= THD_RECT; goto thd2_stuff; }

			if (x1 == thd->pos.x && y1 == thd->pos.y)
				return;
		} else {
			x1 = (x1 + 8) & ~0xF;
			y1 = (y1 + 8) & ~0xF;

			if (thd->flags & THD_RECT) { thd->flags ^= THD_RECT; goto thd3_stuff; }
			if (!(thd->flags & THD_DOT)) { thd->flags ^= THD_DOT; goto thd2_stuff; }

			if (x1 == thd->pos.x && y1 == thd->pos.y)
				return;
		}		
	}

	/* rectangle changed */
	SetSelectionTilesDirty(
		intxchg(thd->pos.x, x1),
		intxchg(thd->pos.y, y1)
	);

	thd->size = thd->to_place_size;
	thd->maxsize = thd->to_place_maxsize;

	SetSelectionTilesDirty(thd->pos.x, thd->pos.y);
	return;

thd3_stuff:
	SetSelectionTilesDirty(thd->pos.x, thd->pos.y);
	thd->maxsize = thd->to_place_maxsize;
	thd->size = thd->to_place_size;
	return;

thd2_stuff:
	thd->size = thd->to_place_size;
	thd->maxsize = thd->to_place_maxsize;
	thd->pos.x = x1;
	thd->pos.y = y1;
	SetSelectionTilesDirty(x1, y1);
	return;
}

void VpStartPlaceSizing(uint tile, int user)
{
	TileHighlightData *thd;

	thd = _thd_ptr;
	thd->userdata = user;
	thd->select_from.x = GET_TILE_X(tile)*16;
	thd->select_to.x = GET_TILE_X(tile)*16;
	thd->select_from.y = GET_TILE_Y(tile)*16;
	thd->select_to.y = GET_TILE_Y(tile)*16;
	thd->place_mode = 3;
	_special_mouse_mode = WSM_SIZING;
}

void VpSetPresizeRange(uint from, uint to)
{
	TileHighlightData *thd = _thd_ptr;
	thd->select_from.x = GET_TILE_X(to)*16;
	thd->select_from.y = GET_TILE_Y(to)*16;
	thd->select_to.x = GET_TILE_X(from)*16;
	thd->select_to.y = GET_TILE_Y(from)*16;
}

void VpStartPreSizing()
{
	_thd.select_from.x = -1;
	_special_mouse_mode = WSM_PRESIZE;
}

void VpSelectTilesWithMethod(int x, int y, int method)
{
	TileHighlightData *thd = _thd_ptr;
	int sx,sy;

	if (x == -1) {
		thd->select_from.x = -1;
		return;
	}

	x &= ~0xF;
	y &= ~0xF;

	sx = thd->select_to.x;
	sy = thd->select_to.y;

	if (method == 1) {
		x = sx;
	} else if (method == 2) {
		y = sy;
	} else {
		if (myabs(sy - y) < myabs(sx - x)) {
			y = sy;
		} else {
			x = sx;
		}
	}
	thd->select_from.x = x;
	thd->select_from.y = y;
}

bool VpHandlePlaceSizingDrag()
{
	Window *w;
	WindowClass wc;
	WindowNumber wn;
	WindowEvent e;
	
	if (_special_mouse_mode != WSM_SIZING)
		return true;

	wc = _thd.window_class;
	wn = _thd.window_number;

	e.place.userdata = _thd.userdata;
	
	if (_left_button_down && (w = FindWindowById(wc,wn)) != NULL) {
		e.event = WE_PLACE_DRAG;
		e.place.pt = GetTileBelowCursor();
		w->wndproc(w, &e);
		return false;
	}
	
	ResetObjectToPlace();

	w = FindWindowById(wc, wn);
	if (!w)
		return false;

	e.event = WE_PLACE_MOUSEUP;
	e.place.pt = _thd.select_from;
	w->wndproc(w, &e);
		
	return false;
}


